from typing import Optional, List, Dict, Any
from ..api import IvrnetClient
from ..models import Environment

def get_client(
    environment: Environment,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> IvrnetClient:
    return IvrnetClient(environment, api_key, username, password, basic_auth)

async def create_voice_file_placeholder(
    filename: str,
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> Dict[str, Any]:
    """Create a voice file placeholder."""
    client = get_client(environment, api_key, username, password, basic_auth)
    payload = {"filename": filename}
    return await client.post("/voice/audio/accesscode/", payload)

async def list_voice_files(
    status: str = "completed", # completed or incomplete
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> List[Dict[str, Any]]:
    """List voice files."""
    client = get_client(environment, api_key, username, password, basic_auth)
    if status == "incomplete":
        return await client.get("/voice/audio/list/accesscode/")
    else:
        return await client.get("/voice/audio/list/")

async def delete_voice_file(
    file_id: str,
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> Dict[str, Any]:
    """Delete a voice file."""
    client = get_client(environment, api_key, username, password, basic_auth)
    return await client.delete(f"/voice/audio/file/{file_id}")
