from typing import Optional, List, Dict, Any
from ..api import IvrnetClient
from ..models import Environment, CreateJobRequest, JobStatus

def get_client(
    environment: Environment,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> IvrnetClient:
    return IvrnetClient(environment, api_key, username, password, basic_auth)

async def create_voice_job(
    from_resource: str,
    to_resource: str,
    languagename: str,
    message: Dict[str, str],
    start_time: str,
    end_time: str,
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
    job_name: Optional[str] = None,
    job_metadata: Optional[Dict[str, str]] = None,
    webhook_url: Optional[str] = None,
    from_resourcename: Optional[str] = None,
    to_resourceid: Optional[str] = None,
    to_resourcename: Optional[str] = None,
    to_variables: Optional[Dict[str, str]] = None,
    to_metadata: Optional[Dict[str, str]] = None,
    voicemail_message: Optional[Dict[str, str]] = None,
    priority: Optional[str] = "normal",
    retry_attempts: Optional[int] = 1,
    retry_interval: Optional[int] = 15,
) -> Dict[str, Any]:
    """Create a voice job."""
    client = get_client(environment, api_key, username, password, basic_auth)
    
    payload = {
        "job": {
            "name": job_name,
            "metadata": job_metadata,
        },
        "from": {
            "resource": from_resource,
            "resourcename": from_resourcename,
        },
        "to": [
            {
                "resourceid": to_resourceid,
                "resource": to_resource,
                "resourcename": to_resourcename,
                "languagename": languagename,
                "variables": to_variables,
                "metadata": to_metadata,
            }
        ],
        "body": {
            "message": message,
            "voicemailmessage": voicemail_message,
        },
        "schedule": {
            "start": start_time,
            "end": end_time,
            "priority": priority,
            "retryattempts": retry_attempts,
            "retryinterval": retry_interval,
        }
    }
    
    if webhook_url:
        payload["job"]["options"] = {"webhook": {"url": webhook_url}}

    return await client.post("/voice/", payload)

async def create_text_job(
    from_resource: str,
    to_resource: str,
    languagename: str,
    message: Dict[str, str],
    start_time: str,
    end_time: str,
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
    job_name: Optional[str] = None,
    job_metadata: Optional[Dict[str, str]] = None,
    webhook_url: Optional[str] = None,
    to_resourceid: Optional[str] = None,
    to_resourcename: Optional[str] = None,
    to_variables: Optional[Dict[str, str]] = None,
    to_metadata: Optional[Dict[str, str]] = None,
    priority: Optional[str] = "normal",
    retry_attempts: Optional[int] = 1,
    retry_interval: Optional[int] = 15,
) -> Dict[str, Any]:
    """Create a text job."""
    client = get_client(environment, api_key, username, password, basic_auth)
    
    payload = {
        "job": {
            "name": job_name,
            "metadata": job_metadata,
        },
        "from": {
            "resource": from_resource,
        },
        "to": [
            {
                "resourceid": to_resourceid,
                "resource": to_resource,
                "resourcename": to_resourcename,
                "languagename": languagename,
                "variables": to_variables,
                "metadata": to_metadata,
            }
        ],
        "body": {
            "message": message,
        },
        "schedule": {
            "start": start_time,
            "end": end_time,
            "priority": priority,
            "retryattempts": retry_attempts,
            "retryinterval": retry_interval,
        }
    }

    if webhook_url:
        payload["job"]["options"] = {"webhook": {"url": webhook_url}}

    return await client.post("/text/", payload)

async def create_email_job(
    from_resource: str,
    to_resource: str,
    languagename: str,
    message: Dict[str, str],
    subject: Dict[str, str],
    start_time: str,
    end_time: str,
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
    job_name: Optional[str] = None,
    job_metadata: Optional[Dict[str, str]] = None,
    webhook_url: Optional[str] = None,
    from_resourcename: Optional[str] = None,
    to_resourceid: Optional[str] = None,
    to_resourcename: Optional[str] = None,
    to_variables: Optional[Dict[str, str]] = None,
    to_metadata: Optional[Dict[str, str]] = None,
    messagetype: Optional[str] = "plaintext",
    priority: Optional[str] = "normal",
    retry_attempts: Optional[int] = 1,
    retry_interval: Optional[int] = 15,
) -> Dict[str, Any]:
    """Create an email job."""
    client = get_client(environment, api_key, username, password, basic_auth)
    
    payload = {
        "job": {
            "name": job_name,
            "metadata": job_metadata,
        },
        "from": {
            "resource": from_resource,
            "resourcename": from_resourcename,
        },
        "to": [
            {
                "resourceid": to_resourceid,
                "resource": to_resource,
                "resourcename": to_resourcename,
                "languagename": languagename,
                "variables": to_variables,
                "metadata": to_metadata,
            }
        ],
        "body": {
            "message": message,
            "subject": subject,
            "options": {
                "messagetype": messagetype
            }
        },
        "schedule": {
            "start": start_time,
            "end": end_time,
            "priority": priority,
            "retryattempts": retry_attempts,
            "retryinterval": retry_interval,
        }
    }

    if webhook_url:
        payload["job"].setdefault("options", {})["webhook"] = {"url": webhook_url}

    return await client.post("/email/", payload)

async def get_job(
    job_id: str,
    job_type: str, # voice, text, email
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> Dict[str, Any]:
    """Get job details."""
    client = get_client(environment, api_key, username, password, basic_auth)
    return await client.get(f"/{job_type}/{job_id}")

async def list_jobs(
    job_type: str, # voice, text, email
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
    page: int = 1,
    page_size: int = 10,
    status: JobStatus = JobStatus.ALL,
) -> List[Dict[str, Any]]:
    """List jobs."""
    client = get_client(environment, api_key, username, password, basic_auth)
    params = {
        "page": page,
        "pagesize": page_size,
        "status": status.value,
    }
    return await client.get(f"/{job_type}/list/", params=params)

async def cancel_job(
    job_id: str,
    job_type: str, # voice, text, email
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> Dict[str, Any]:
    """Cancel a job."""
    client = get_client(environment, api_key, username, password, basic_auth)
    return await client.delete(f"/{job_type}/{job_id}")

async def pause_resume_job(
    job_id: str,
    job_type: str, # voice, text, email
    action: str, # pause or resume
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> Dict[str, Any]:
    """Pause or resume a job."""
    client = get_client(environment, api_key, username, password, basic_auth)
    payload = {
        "jobid": job_id,
        "action": action
    }
    return await client.put(f"/{job_type}/", payload)

async def send_followup_message(
    job_id: str,
    resource: str,
    message: str,
    environment: Environment = Environment.PRODUCTION_CA,
    api_key: Optional[str] = None,
    username: Optional[str] = None,
    password: Optional[str] = None,
    basic_auth: Optional[str] = None,
) -> Dict[str, Any]:
    """Send a follow-up text message."""
    client = get_client(environment, api_key, username, password, basic_auth)
    payload = {
        "jobid": job_id,
        "resource": resource,
        "message": message
    }
    return await client.post("/text/msg/recipient/", payload)
