from enum import Enum
from typing import List, Optional, Dict, Any, Union
from pydantic import BaseModel, Field

class Environment(str, Enum):
    PRODUCTION_CA = "production-ca"
    PRODUCTION_US = "production-us"
    TEST = "test"

class JobStatus(str, Enum):
    SCHEDULED = "scheduled"
    RUNNING = "running"
    COMPLETED = "completed"
    EXPIRED = "expired"
    PAUSED = "paused"
    CANCELLED = "cancelled"
    ALL = "all"

class Priority(str, Enum):
    EMERGENCY = "emergency"
    URGENT = "urgent"
    NORMAL = "normal"
    LOW = "low"
    TEST = "test"

class MessageType(str, Enum):
    TTS = "tts"
    VOICE = "voice"
    HYBRID = "hybrid"
    PLAINTEXT = "plaintext"
    HTML = "html"

class WebhookType(BaseModel):
    scheduled: Optional[bool] = None
    running: Optional[bool] = None
    completed: Optional[bool] = None
    expired: Optional[bool] = None

class ResponseVariables(BaseModel):
    # Dynamic keys, but we can define common ones or use Dict
    pass

class WebhookTypes(BaseModel):
    jobstatus: Optional[WebhookType] = None
    responsevariables: Optional[Dict[str, bool]] = None

class Retry(BaseModel):
    attempts: Optional[int] = 3
    interval: Optional[int] = 15
    interval_list: Optional[List[int]] = None
    statuscode_list: Optional[List[int]] = None

class WebhookAuth(BaseModel):
    apikey: Optional[str] = None
    username: Optional[str] = None
    password: Optional[str] = None
    basicauth: Optional[str] = None

class Webhook(BaseModel):
    url: str
    auth: Optional[WebhookAuth] = None
    method: Optional[str] = "POST"
    retry: Optional[Retry] = None
    types: Optional[WebhookTypes] = None

class JobOptions(BaseModel):
    webhook: Optional[Webhook] = None

class FromResource(BaseModel):
    resource: str
    resourcename: Optional[str] = None

class ToResource(BaseModel):
    resourceid: Optional[str] = None
    resource: str
    resourcename: Optional[str] = None
    languagename: str
    variables: Optional[Dict[str, str]] = None
    metadata: Optional[Dict[str, str]] = None

class MessageVariableTags(BaseModel):
    opentag: str
    closetag: str

class EmailAttachment(BaseModel):
    filename: str
    filebase64data: str

class BodyOptions(BaseModel):
    messagetype: Optional[MessageType] = None
    messagevariabletags: Optional[MessageVariableTags] = None
    ttsmessagevoice: Optional[str] = None # female or male
    responsemessageprompt: Optional[Dict[str, str]] = None
    responsesuccessmessage: Optional[Dict[str, str]] = None
    responseerrormessage: Optional[Dict[str, str]] = None
    responsevariables: Optional[Dict[str, str]] = None
    responsevariablesactions: Optional[Dict[str, str]] = None
    responsevariablesmessages: Optional[Dict[str, Dict[str, str]]] = None
    hangupmessage: Optional[Dict[str, str]] = None
    transfermessage: Optional[Dict[str, str]] = None
    jumpmessage: Optional[Dict[str, str]] = None
    languagetranslate: Optional[bool] = False
    emailattachments: Optional[List[EmailAttachment]] = None

class Body(BaseModel):
    message: Dict[str, str]
    voicemailmessage: Optional[Dict[str, str]] = None
    subject: Optional[Dict[str, str]] = None # For email
    options: Optional[BodyOptions] = None

class Schedule(BaseModel):
    priority: Optional[Priority] = Priority.NORMAL
    start: str # ISO8601
    end: str # ISO8601
    retryattempts: Optional[int] = None
    retryinterval: Optional[int] = None

class CreateJobRequest(BaseModel):
    job: Optional[Dict[str, Any]] = None # name, metadata, options
    from_: FromResource = Field(..., alias="from")
    to: List[ToResource]
    body: Body
    schedule: Schedule

class JobResponse(BaseModel):
    jobid: Optional[str] = None
    payloadResponse: Optional[Dict[str, Any]] = None
    payloadValidation: Optional[Dict[str, Any]] = None

class VoiceFileResponse(BaseModel):
    tollfree: Optional[str] = None
    accesscode: Optional[str] = None
    fileid: Optional[str] = None
    filename: Optional[str] = None
