import httpx
from typing import Optional, Dict, Any
from .models import Environment

ENV_URLS = {
    Environment.PRODUCTION_CA: "https://message.ivrnet.com/api/v1",
    Environment.PRODUCTION_US: "https://message.us.ivrnet.com/api/v1",
    Environment.TEST: "https://message-staging.ivrnet.com/api/v1",
}

class IvrnetClient:
    def __init__(
        self,
        environment: Environment = Environment.PRODUCTION_CA,
        api_key: Optional[str] = None,
        username: Optional[str] = None,
        password: Optional[str] = None,
        basic_auth: Optional[str] = None,
    ):
        self.base_url = ENV_URLS[environment]
        self.headers = {"Content-Type": "application/json"}
        
        if api_key:
            self.headers["X-API-KEY"] = api_key
        elif basic_auth:
            self.headers["Authorization"] = basic_auth
        elif username and password:
            import base64
            credentials = f"{username}:{password}"
            encoded = base64.b64encode(credentials.encode()).decode()
            self.headers["Authorization"] = f"Basic {encoded}"
        else:
            raise ValueError("Authentication is required: provide api_key or username/password or basic_auth")

    async def request(self, method: str, endpoint: str, data: Optional[Dict[str, Any]] = None, params: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        url = f"{self.base_url}{endpoint}"
        async with httpx.AsyncClient() as client:
            response = await client.request(method, url, headers=self.headers, json=data, params=params)
            try:
                response.raise_for_status()
            except httpx.HTTPStatusError as e:
                # Try to parse error message from body
                try:
                    error_body = response.json()
                    raise Exception(f"API Error: {e.response.status_code} - {error_body}")
                except:
                    raise e
            
            if response.status_code == 204:
                return {}
            return response.json()

    async def post(self, endpoint: str, data: Dict[str, Any]) -> Dict[str, Any]:
        return await self.request("POST", endpoint, data=data)

    async def get(self, endpoint: str, params: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        return await self.request("GET", endpoint, params=params)

    async def put(self, endpoint: str, data: Dict[str, Any]) -> Dict[str, Any]:
        return await self.request("PUT", endpoint, data=data)

    async def delete(self, endpoint: str, data: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        return await self.request("DELETE", endpoint, data=data)
